/* -*-c++-*- 
 * This source code is proprietary of ADIT
 * Copyright (C) 2013 Advanced Driver Information Technology Joint Venture GmbH
 * All rights reserved
 *
 * Author: Vadiraj Kaamsha <vadiraj.kaamsha@in.bosch.com>
 * Author: Rudolf Dederer <rudolf.dederer@de.bosch.com>
*/

#include <osgBatchedText/GlyphInfoContainer>
#include <osgBatchedText/BatchElementBase>
#include <osgBatchedText/BatcherBase>

using namespace osgBatchedText;

GlyphInfoContainerBase::GlyphInfoContainerBase(const FontDescr& fontDescr, bool outline, unsigned int maxGlyphs) :
   _font(NULL),
   _kerningType(osgText::KERNING_NONE),
   _hasKerning(false),
   _outline(outline),
   _fontSize(32u),
   _strokeWidth(2u),
   _maxGlyphs(maxGlyphs),
   _counter(0),
   _lineBreakCharCodes(' ', '-', '/'),
   _lineBreakGlyphIndices(INVALID_GLYPH_INDEX, INVALID_GLYPH_INDEX, INVALID_GLYPH_INDEX)
{
   setFont(fontDescr._font, fontDescr._fontSize, fontDescr._outlineSize);
   _blacklistedGlyphIndices.reserve(_maxGlyphs);

   osgText::Font* activefont = getActiveFont();
   if (activefont)
   {
      unsigned int lineBreakGlyphIndex = INVALID_GLYPH_INDEX;
      if (activefont->getGlyphIndex(lineBreakGlyphIndex, _lineBreakCharCodes._space))
      {
         _lineBreakGlyphIndices._space = lineBreakGlyphIndex;
      }
      if (activefont->getGlyphIndex(lineBreakGlyphIndex, _lineBreakCharCodes._hyphen))
      {
         _lineBreakGlyphIndices._hyphen = lineBreakGlyphIndex;
      }
      if (activefont->getGlyphIndex(lineBreakGlyphIndex, _lineBreakCharCodes._slash))
      {
         _lineBreakGlyphIndices._slash = lineBreakGlyphIndex;
      }
   }
}

GlyphInfoContainerBase::~GlyphInfoContainerBase()
{
   clearGlyphMap();
}

void GlyphInfoContainerBase::clearGlyphMap()
{
   OpenThreads::ScopedLock<OpenThreads::Mutex> lock(_glyphInfoContainerMutex);

   for (std::list<BatchEntry*>::iterator gitr = _glyphContainer.begin(); gitr != _glyphContainer.end(); ++gitr)
   {
      delete *gitr;
   }

   _glyphMapOfGlyphIndices.clear();
   _glyphMapOfCharCodes.clear();
   _glyphContainer.clear();
}

void GlyphInfoContainerBase::setFont(const osg::ref_ptr<osgText::Font>& font, unsigned int fontSize, unsigned int strokeWidth)
{
   if ((_font.get() != font.get()) || (_fontSize != fontSize) || (_strokeWidth != strokeWidth))
   {
      clearGlyphMap();
   }

   if (_font.get() != font.get())
   {
      _font = font;
   }

   if (_font.valid())
   {
      _font->setFontResolution(osgText::FontResolution(fontSize, fontSize));
      _hasKerning = _font->hasKerning();
      if (_hasKerning)
      {
         _kerningType = osgText::KERNING_DEFAULT;
      }
   }
   _strokeWidth = strokeWidth;
}

void GlyphInfoContainerBase::constructDefaultGlyphs(const unsigned int* additionalCharCodes, unsigned int numCharCodes)
{
   ++_counter;
   static unsigned int charCodes[] = { //ASCII
      0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, /*0-9*/
      0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a,
      0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54,
      0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, /*A-Z*/
      0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a,
      0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74,
      0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, /*a-z*/
      0x20, /* (Blank) */
      0x2d, /* - */
      0x2e, /* . */
      0x2f  /* / */
   };

   OpenThreads::ScopedLock<OpenThreads::Mutex> lock(_glyphInfoContainerMutex);
   for(unsigned int cntr = 0; cntr < sizeof(charCodes) / 4u; ++cntr)
   {
      addGlyphToCache(true, charCodes[cntr]);
   }
   if (additionalCharCodes != NULL)
   {
      for (unsigned int cntr = 0; cntr < numCharCodes; ++cntr)
      {
         addGlyphToCache(true, additionalCharCodes[cntr]);
      }
   }
}

bool GlyphInfoContainerBase::getGlyph(bool isCharCode, const unsigned int charCodeOrGlyphIndex, osgText::glyphEntryUpper& upper, unsigned int& prevglyphindex, osg::Vec2& delta, bool isRightToLeft, bool readKerning)
{
   ++_counter;
   osgText::Font* activefont = getActiveFont();

   {
      OpenThreads::ScopedLock<OpenThreads::Mutex> lock(_glyphInfoContainerMutex);
      BatchEntry* batchEntry = (activefont == NULL) ? NULL : addGlyphToCache(isCharCode, charCodeOrGlyphIndex);
      if (batchEntry)
      {
         upper = batchEntry->_glyph._upper;
      }
      else
      {
         return false;
      }
   }

   if (readKerning)
   {
      if ((prevglyphindex == 0xFFFFFFFF) || (!_hasKerning) || (_kerningType == osgText::KERNING_NONE))
      {
         delta = osg::Vec2();
      }
      else
      {
         if (isRightToLeft)
         {
            delta = activefont->getKerning(upper._glyphIndex, prevglyphindex, _kerningType, true);
         }
         else
         {
            delta = activefont->getKerning(prevglyphindex, upper._glyphIndex, _kerningType, true);
         }
      }
   }
   prevglyphindex = upper._glyphIndex;

   return true;
}

BatchEntry* GlyphInfoContainerBase::addGlyphToCache(bool isCharCode, const unsigned int charCodeOrGlyphIndex) const
{
   BatchEntry* entry = NULL;

   if (isCharCode)
   {
      GlyphMap::iterator gmccItr = _glyphMapOfCharCodes.find(charCodeOrGlyphIndex);
      if (gmccItr != _glyphMapOfCharCodes.end())
      {
         entry = gmccItr->second;
         gmccItr->second->_counter = _counter;
      }
   }

   if (!entry)
   {
      osgText::Font* activefont = getActiveFont();
      if (activefont)
      {
         unsigned int glyphIndex = 0;
         if (isCharCode)
         {
            activefont->getGlyphIndex(glyphIndex, charCodeOrGlyphIndex);
         }
         else
         {
            glyphIndex = charCodeOrGlyphIndex;
         }

         GlyphMap::iterator gmgiItr = _glyphMapOfGlyphIndices.find(glyphIndex);
         if (gmgiItr != _glyphMapOfGlyphIndices.end())
         {
            entry = gmgiItr->second;
            gmgiItr->second->_counter = _counter;
            if (isCharCode)
            {
               gmgiItr->second->_charcode = charCodeOrGlyphIndex;
               gmgiItr->second->_isCharCodeValid = true;
               _glyphMapOfCharCodes[charCodeOrGlyphIndex] = entry;
            }
         }
         else
         {
            if (_glyphContainer.size() == _maxGlyphs)
            {
               /* to many glyphs on hold so remove some which are not in use ....*/
               //sort
               _glyphContainer.sort(BatchEntry::timeCompare);
               //remove oldest, vjk1kor- how many should we remove?
               std::list<BatchEntry*>::iterator itr = _glyphContainer.begin();
               if (itr != _glyphContainer.end())
               {
                  GlyphMap::iterator itrMap = _glyphMapOfCharCodes.find((*itr)->_charcode);
                  if (_glyphMapOfCharCodes.end() != itrMap)
                  {
                     // Remove if the glyph is also contained in the _glyphMapOfCharCodes
                     // All entries in _glyphContainer are contained in _glyphMapOfGlyphIndices(and vice-versa) but not necessarily in _glyphMapOfCharCodes
                     _glyphMapOfCharCodes.erase(itrMap);
                  }
                  _glyphMapOfGlyphIndices.erase((*itr)->_glyph._upper._glyphIndex);

                  addForDeletion(*itr);
                  _glyphContainer.erase(itr);
               }
            }

            if ((_glyphContainer.size() < _maxGlyphs) && !isBlackListedGlyphIndex(glyphIndex))
            {
               entry = isCharCode ? new BatchEntry(charCodeOrGlyphIndex) : new BatchEntry;
               if (NULL != entry)
               {
                  /* glyph is not in map -> get it from freetype */
                  if (activefont->getGlyphForGlyphIndex(entry->_glyph, glyphIndex, _outline, _strokeWidth))
                  {
                     entry->_counter = _counter;
                     if (isCharCode)
                     {
                        _glyphMapOfCharCodes[charCodeOrGlyphIndex] = entry;
                     }

                     _glyphMapOfGlyphIndices[glyphIndex] = entry;
                     _glyphContainer.push_back(entry);

                     _newGlyphIndices.push_back(glyphIndex);
                  }
                  else
                  {
                     //Trace error
                     if (isCharCode)
                     {
                        OSG_WARN << __LINE__ << __FILE__ << "Could not create glyph for char code " << charCodeOrGlyphIndex << std::endl;
                     }
                     else
                     {
                        OSG_WARN << __LINE__ << __FILE__ << "Could not create glyph for glyph index " << glyphIndex << std::endl;
                     }

                     _blacklistedGlyphIndices.push_back(glyphIndex);

                     delete entry;
                     entry = NULL;
                  }
               }
            }
         }
      }
      else
      {
         //Trace error
         OSG_WARN << __LINE__ << __FILE__ << "No active Font set" << std::endl;
      }
   }

   return entry;
}

bool GlyphInfoContainerBase::hasNewGlyphs() const
{
   return !_newGlyphIndices.empty();
}

void GlyphInfoContainerBase::getNewGlyphs(std::vector<BatchEntry*>& glyphs, bool shallowCopy)
{
   glyphs.clear();
   glyphs.reserve(_newGlyphIndices.size());
   OpenThreads::ScopedLock<OpenThreads::Mutex> lock(_glyphInfoContainerMutex);
   for (std::vector<unsigned int>::iterator itr = _newGlyphIndices.begin(); itr != _newGlyphIndices.end(); ++itr)
   {
      GlyphMap::iterator gitr = _glyphMapOfGlyphIndices.find(*itr);
      if (gitr != _glyphMapOfGlyphIndices.end())
      {
         glyphs.push_back(new BatchEntry(*(gitr->second), _outline, shallowCopy));
         if (shallowCopy)
         {
            //remove data from original container
            gitr->second->_glyph._lower._data = NULL;
         }
      }
   }
   _newGlyphIndices.clear();
}


void GlyphInfoContainerBase::addNewGlyphs(std::vector<BatchEntry*>& glyphs)
{
   OpenThreads::ScopedLock<OpenThreads::Mutex> lock(_glyphInfoContainerMutex);
   for (std::vector<BatchEntry*>::iterator itr = glyphs.begin(); itr != glyphs.end(); ++itr)
   {
      if (*itr != NULL)
      {
         unsigned int glyphIndex = (*itr)->_glyph._upper._glyphIndex;
         GlyphMap::iterator gitr = _glyphMapOfGlyphIndices.find(glyphIndex);
         if (gitr != _glyphMapOfGlyphIndices.end())
         {
            delete *itr;
         }
         else
         {
            _glyphMapOfGlyphIndices[glyphIndex] = *itr;
            if ((*itr)->_isCharCodeValid)
            {
               _glyphMapOfCharCodes[(*itr)->_charcode] = *itr;
            }

            _glyphContainer.push_back(*itr);
         }
         *itr = NULL;
      }
   }
}

bool GlyphInfoContainerBase::getActiveFontResolution(osgText::FontResolution& fontSize) const
{
   bool success = false;

   osgText::Font* activeFont = getActiveFont();
   if (activeFont)
   {
      success = activeFont->getFontResolution(fontSize);
   }

   return success;
}

BatchEntry::BatchEntry(const BatchEntry& src, bool outline, bool shallowCopy)
 : _charcode(src._charcode),
   _counter(src._counter),
   _isCharCodeValid(src._isCharCodeValid),
   _glyph(src._glyph)
{
   if (!shallowCopy)
   {
      unsigned int width = static_cast<unsigned int>(_glyph._upper._dimension.x());
      unsigned int height = static_cast<unsigned int>(-_glyph._upper._dimension.y());
      unsigned int dataSize = width * height * (outline ? 2u : 1u);

      _glyph._lower._data = new unsigned char[dataSize];
      for (unsigned int loop = 0; loop < height; loop++)
      {
         memcpy(_glyph._lower._data, src._glyph._lower._data, dataSize);
      }
   }
}

void BatchEntry::clearGlyphVector(std::vector<BatchEntry*>& glyphs)
{
   for (std::vector<BatchEntry*>::iterator itr = glyphs.begin(); itr != glyphs.end(); ++itr)
   {
      delete *itr;
   }
   glyphs.clear();
}


GlyphInfoContainer::GlyphInfoContainer(const FontDescr& fontDescr, bool outline, unsigned int maxGlyphs)
   : GlyphInfoContainerBase(fontDescr, outline, maxGlyphs)
{
}

